/******************************************************/
/* Examples Program For "ET-BASE dsPIC30F2010"  Board */
/* Hardware    : ET-BASE dsPIC30F2010                 */
/* Target MCU  : dsPIC30F2010                         */
/*       	   : X-TAL : 7.3728 MHz                   */
/*             : Run 117.9648MHz                      */
/*             : Selec OSC Mode = XT w/PLL 16x        */
/* Compiler    : MPLAB v8.40 + C30 v3.20B      	      */
/* Last Update : 12/October/2009                      */
/******************************************************/
/* Function    : Example Use SPI Connect MCP4922      */
/*             : MCP4922 Generate Sinewave & Saw      */	
/*             : DAC-A Generate Sinewave              */
/*             : DAC-B Generate Saw                   */
/******************************************************/
/* Used ET-MINI MCP4922 Module Interface              */
/* SDI   = RF3(SPI-SDO1)                              */
/* CS#   = RB2(SPI-SS#1)                              */
/* SCK   = RE8(SPI-SCK1)                              */
/* LDAC# = RB0                                        */
/* SHDN# = RB1                                        */
/******************************************************/

#include <p30f2010.h>                                		// dsPIC30F2010 MPU Register
#include "spi.h"											// Used SPI Function Library

/* Setup Configuration For ET-BASE dsPIC30F2010 */
_FOSC(CSW_FSCM_OFF & XT_PLL16);								// Disable Clock Switching,Enable Fail-Salf Clock
                                                            // Clock Source = Primary XT + (PLL x 16)
_FWDT(WDT_OFF);												// Disable Watchdog 
_FBORPOR(PBOR_OFF & PWRT_64 & MCLR_EN);					    // Disable Brown-Out ,Power ON = 64mS,Enable MCLR
_FGS(CODE_PROT_OFF);										// Code Protect OFF
/* End   Configuration For ET-BASE dsPIC30F2010 */

// Array[0..64] = 64 Step = Degree[0..360]/64 = 5.625/Step
// Step Degree[0..63] = 0,5.625,11.25,16.875,...,354.375,360
// Step Output[0..63] = (sin(x)*2048)+2048
// Example = (sin(5.625)*2048)+2048
//         = (0.098*2048)+2048
//         = 2248.739 
//         = 2048 
//         = 0x08C8
// Sine Function Table
unsigned int table_sine[64] = {
 	        					0x07FF, 0x08C8, 0x098E, 0x0A51, 	
		    					0x0B0F, 0x0BC4, 0x0C71, 0x0D12,
 	        					0x0DA7, 0x0E2E, 0x0EA5, 0x0F0D, 
		    					0x0F63, 0x0FA6, 0x0FD7, 0x0FF5,
 		    					0x0FFF, 0x0FF5, 0x0FD7, 0x0FA6, 
		    					0x0F63, 0x0F0D, 0x0EA5, 0x0E2E,
 	        					0x0DA7, 0x0D12, 0x0C71, 0x0BC4, 
		    					0x0B0F, 0x0A51, 0x098E, 0x08C8,
 	        					0x07FF, 0x0736, 0x0670, 0x05AD, 
		    					0x04EF, 0x043A, 0x038D, 0x02EC,
 	        					0x0257, 0x01D0, 0x0159, 0x00F1, 
		    					0x009B, 0x0058, 0x0027, 0x0009,
 	        					0x0000, 0x0009, 0x0027, 0x0058, 
		    					0x009B, 0x00F1, 0x0159, 0x01D0,
 	        					0x0257, 0x02EC, 0x038D, 0x043A, 
		    					0x04EF, 0x05AD, 0x0670, 0x0736  
	      					  };

/* pototype  section */
void delay(unsigned long int);							    // Delay Time Function
void init_spi(void);										// Initial SPI For Interface MCP4922

int main(void)
{  
  int  sine_cnt;
  unsigned int sine_data;

  TRISBbits.TRISB0 = 0;										// Config RB0 = Output(LDAC#)
  TRISBbits.TRISB1 = 0;										// Config RB1 = Output(SHDN#)
  TRISBbits.TRISB2 = 0;										// Config RB2 = Output(CS#)

  init_spi();												// Initial SPI Function
  LATBbits.LATB1 = 1;									    // Active MCP4922(SHDN# = 1)
 
  // Start Test Output SPI(MCP4922) //
  while(1)													// Loop Continue
  {
    for(sine_cnt=0;sine_cnt<64;sine_cnt++)					// Loop Get Array Sine Table 0-63
    {      
      //Generate Sinewave on DAC-A
      LATBbits.LATB2 = 0;									// Enable SPI Chips Select

      sine_data = table_sine[sine_cnt];
      sine_data |= 0x3000;									// Write DAC-A
      WriteSPI1(sine_data);			  	    				// Send Data to MCP4922     
	  while(SPI1STATbits.SPITBF);							// Wait SPI Send Complete	
	
      LATBbits.LATB2 = 1;									// Disable SPI Chips Select

      LATBbits.LATB0 = 0;									// Latch DAC Output(LDAC# = 0)
	  LATBbits.LATB0 = 1;									// LDAC# = 1  			   	    

      //Generate Saw on DAC-B
      LATBbits.LATB2 = 0;									// Enable SPI Chips Select

      sine_data = sine_cnt*64;
      sine_data |= 0xB000;									// Write DAC-B
      WriteSPI1(sine_data);			  	    				// Send Data to MCP4922     
	  while(SPI1STATbits.SPITBF);							// Wait SPI Send Complete	
	
      LATBbits.LATB2 = 1;									// Disable SPI Chips Select

      LATBbits.LATB0 = 0;									// Latch DAC Output(LDAC# = 0)
	  LATBbits.LATB0 = 1;									// LDAC# = 1 
	} 	    
  }   							 
}	  

/*************************************/
/* Initial dsPIC30F2010 SPI Function */
/* For Interface MCP4922(SPI DAC)    */
/*************************************/
void init_spi(void)
{
  CloseSPI1();												// Close SPI Before New Config
  
  // Initial SPI Interrupt Control
  ConfigIntSPI1(SPI_INT_DIS &								// Disable SPI Interrupt
                SPI_INT_PRI_7);								// Set Priority Interrupt = 7

  //***************************************************
  // ET-BASE dsPIC30F2010 Hardware Board
  // XTAL = 7.3728MHz
  // Fosc = 7.3728 MHz x 16 = 117.9648 MHz
  // Fcy  = Fosc / 4 
  //      = 117.9648 / 4 = 29.4912 MHz
  // Tcy  = 1 / 29.4912 MHz
  //      = 33.90842 nS
  //***************************************************
  // Open SPI Function (460.8KHz Baud)
  OpenSPI1(FRAME_ENABLE_OFF &								// Disable SPI Frame
           FRAME_SYNC_OUTPUT &								// SPI = Output
           ENABLE_SDO_PIN &									// SDO = Output Data to SDI of MCP4922
		   SPI_MODE16_ON &									// SPI = Word Mode(16 Bit Data)
           SPI_SMP_ON &										// Sampling Data on Half of Data Clock
		   SPI_CKE_ON &										// Change Data on Falling Edge Clock	
		   SLAVE_ENABLE_OFF &								// Disable SS# Pin Function
		   CLK_POL_ACTIVE_HIGH &							// Rising Edge Clock Active
		   MASTER_ENABLE_ON &								// SPI Function = Master
		   SEC_PRESCAL_4_1 &								// Secondary Prescale = 4:1
		   PRI_PRESCAL_16_1 ,								// Primary Prescale = 16:1
           
           SPI_ENABLE &										// Enable SPI Function
		   SPI_IDLE_STOP &									// Disable SPI in IDLE Mode
		   SPI_RX_OVFLOW_CLR);								// Clear Overflow Flag
}
